// modules/aiimageedit/views/js/ai_image_tools.js

/**
 * Upscales an image client-side using canvas if it's smaller than the original product image dimensions.
 * Tries to match the original dimensions by applying a multiplier, but won't exceed original dimensions.
 * Preserves the aspect ratio of the source (AI generated) image.
 *
 * @param {string} aiImageUrl The URL of the AI-generated image to upscale.
 * @param {number} originalProductWidth The width of the original product image.
 * @param {number} originalProductHeight The height of the original product image.
 * @param {number} upscaleMultiplier The desired upscale factor (e.g., 1.5 for 150%, 2 for 200%).
 * @returns {Promise<string>} A promise that resolves with the base64 data URL of the upscaled image,
 *                            or the original aiImageUrl if no upscaling was needed or possible.
 *                            Rejects on error.
 */
async function upscaleImageAIClientSide(aiImageUrl, originalProductWidth, originalProductHeight, upscaleMultiplier = 1.0) {
    console.log("[Upscaler] Starting upscale process for:", aiImageUrl);
    console.log("[Upscaler] Original Product Dimensions:", originalProductWidth, "x", originalProductHeight);
    console.log("[Upscaler] Desired Multiplier:", upscaleMultiplier);

    return new Promise((resolve, reject) => {
        const aiImage = new Image();
        

        aiImage.onload = () => {
            console.log("[Upscaler] AI Image loaded. Dimensions:", aiImage.naturalWidth, "x", aiImage.naturalHeight);

            let currentAiWidth = aiImage.naturalWidth;
            let currentAiHeight = aiImage.naturalHeight;

            // Only upscale if the AI image is smaller in at least one dimension than the original
            // AND the multiplier suggests an increase.
            if (upscaleMultiplier <= 1.0 && (currentAiWidth >= originalProductWidth || currentAiHeight >= originalProductHeight)) {
                console.log("[Upscaler] No upscale needed (AI image large enough or multiplier <= 1). Returning original URL.");
                resolve(aiImageUrl); // Return original URL, not data URL, to keep consistency if no change
                return;
            }
            if (currentAiWidth === 0 || currentAiHeight === 0) {
                 console.error("[Upscaler] AI image has zero dimensions. Cannot upscale.");
                 resolve(aiImageUrl); // Cannot process, return original
                 return;
            }

            // Calculate potential new dimensions based on multiplier
            let targetWidth = currentAiWidth * upscaleMultiplier;
            let targetHeight = currentAiHeight * upscaleMultiplier;
            console.log("[Upscaler] Potential target dimensions after multiplier:", targetWidth, "x", targetHeight);

            // Ensure target dimensions do not exceed original product image dimensions
            // while maintaining the AI image's aspect ratio.
            const aspectRatio = currentAiWidth / currentAiHeight;

            if (targetWidth > originalProductWidth) {
                targetWidth = originalProductWidth;
                targetHeight = targetWidth / aspectRatio;
            }
            if (targetHeight > originalProductHeight) {
                targetHeight = originalProductHeight;
                targetWidth = targetHeight * aspectRatio;
            }
            
            // Final check: if calculated target is still smaller than current AI image (e.g. due to cap by very small original)
            // or if multiplier effectively made it smaller, don't downscale.
            if (targetWidth < currentAiWidth || targetHeight < currentAiHeight) {
                 console.log("[Upscaler] Calculated target dimensions are smaller than current AI image after capping. No upscale performed.");
                 resolve(aiImageUrl);
                 return;
            }


            targetWidth = Math.round(targetWidth);
            targetHeight = Math.round(targetHeight);

            // If after all calculations, the target is not significantly larger, don't bother.
            if (targetWidth <= currentAiWidth && targetHeight <= currentAiHeight) {
                 console.log("[Upscaler] Target dimensions are not larger than current AI image. No upscale performed.");
                 resolve(aiImageUrl);
                 return;
            }


            console.log("[Upscaler] Final Target Upscaled Dimensions:", targetWidth, "x", targetHeight);

            const canvas = document.createElement('canvas');
            canvas.width = targetWidth;
            canvas.height = targetHeight;
            const ctx = canvas.getContext('2d');

            // Setting imageSmoothingQuality can sometimes help, but browser support varies.
            // Values: "low", "medium", "high"
            if (ctx.imageSmoothingQuality) {
                ctx.imageSmoothingQuality = 'high';
            }
            ctx.imageSmoothingEnabled = true; // Ensure smoothing is on

            console.log("[Upscaler] Drawing image to canvas for upscaling...");
            ctx.drawImage(aiImage, 0, 0, targetWidth, targetHeight);

            try {
                // It's better to get the format from the original AI image if possible,
                // or from the 'output_format' selected by the user.
                // For simplicity, let's assume JPEG output for upscaled images here.
                const upscaledDataUrl = canvas.toDataURL('image/jpeg', 0.92); // 0.92 is a good quality for JPEG
                console.log("[Upscaler] Upscaling successful. Returning new data URL.");
                resolve(upscaledDataUrl);
            } catch (e) {
                console.error("[Upscaler] Error converting canvas to data URL:", e);
                reject(new Error('Failed to convert upscaled image to data URL.'));
            }
        };

        aiImage.onerror = (e) => {
            console.error("[Upscaler] Error loading AI image from base64 data URL for upscaling. Error:", e);
            reject(new Error('Failed to load AI image from base64 data for upscaling.'));
        };

        aiImage.src = aiImageBase64DataUrl; // Source is now the base64 data URI
    });
}

// Example of how you might expose it if needed (not strictly necessary if called directly)
// window.AIImageTools = {
//  upscaleImageAIClientSide: upscaleImageAIClientSide
// };