<?php
/**
 * 2007-2023 PrestaShop
 *
 * NOTICE OF LICENSE
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    AISyncrasy
 * @copyright 2007-2025 Modasyncrasy S.L.
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 */

if (!defined('_PS_VERSION_')) {
    exit;
}


class AdminAiImageEditController extends ModuleAdminController
{
    public function __construct()
    {
        $this->bootstrap = true;
		$this->module = Module::getInstanceByName('aiimageedit'); // Add this line
        parent::__construct();
        // $this->className = 'Configuration'; // Not managing a specific PS ObjectModel here
        // $this->table = 'configuration'; // Not directly managing ps_configuration
    }

    /**
     * Helper for consistent JSON responses.
     */
    protected function ajaxDieJson($data)
    {
        header('Content-Type: application/json');
        die(json_encode($data));
    }

  public function ajaxProcessGetAccountStatus()
    {
        $id_shop = (int)$this->context->shop->id;
        $apiKey = Configuration::get('AIIMAGEEDIT_API_KEY_BFL', null, null, $id_shop);

        if (empty($apiKey)) {
            $this->ajaxDieJson(['success' => false, 'error' => 'No API Key']);
        }
        
        // Usamos el mismo método privado que ya tienes en aiimageedit.php.
        // Para hacerlo accesible, podrías moverlo a un helper o duplicar la lógica aquí.
        // Por simplicidad, duplicaremos la llamada cURL aquí.
        
        $curl = curl_init();
	  	if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/account/details',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => ['x-api-key: ' . $apiKey],
        ]);
        
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300 && isset($responseData['id'])) {
            // Guardamos el ID por si acaso no se ha guardado antes
            Configuration::updateValue('AIIMAGEEDIT_ACCOUNT_ID', $responseData['id']);
            $this->ajaxDieJson(['success' => true, 'account_details' => $responseData]);
        } else {
            $this->ajaxDieJson(['success' => false, 'error' => 'Failed to fetch details']);
        }
    }
	
	 /**
     * AJAX action to change the user's account email.
     */
    public function ajaxProcessChangeEmail()
    {
        $id_shop = (int)$this->context->shop->id;
        $apiKey = Configuration::get('AIIMAGEEDIT_API_KEY_BFL', null, null, $id_shop);
        $newEmail = Tools::getValue('newEmail');

        if (empty($apiKey)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('API Key is not configured.')]);
        }
        
        if (empty($newEmail) || !Validate::isEmail($newEmail)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('The provided email is not valid.')]);
        }

        $payload = json_encode(['newEmail' => $newEmail]);

        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/account/change-email',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'x-api-key: ' . $apiKey,
            ],
        ]);
        
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300 && isset($responseData['success']) && $responseData['success'] === true) {
            // Asumimos que la API devuelve {success: true, message: "..."}
            $this->ajaxDieJson([
                'success' => true, 
                'message' => $responseData['message'] ?? $this->module->l('Email updated successfully.'),
                'newEmail' => $newEmail // Devolvemos el nuevo email para actualizar la UI
            ]);
        } else {
            $errorMessage = $responseData['message'] ?? $this->module->l('Could not update email.');
            $this->ajaxDieJson(['success' => false, 'message' => $errorMessage]);
        }
    }

    /**
     * AJAX action to register a new user with an email and get an API key.
     */
    public function ajaxProcessRegisterUser()
    {
        $domain_full = Tools::getShopDomainSSL(true, true);
        $domain = preg_replace('#^https?://#', '', rtrim($domain_full, '/'));
        $userEmail = Tools::getValue('email');
        $currentLanguage = new Language($this->context->language->id);
        $lang_iso_code = $currentLanguage->iso_code;
		
        if (empty($userEmail) || !Validate::isEmail($userEmail)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Please provide a valid email address.')]);
        }

        $payload_array = [
            'origin' => 'prestashop',
            'domain' => $domain,
            'email' => $userEmail,
			'lang' => $lang_iso_code,
        ];
        $payload = json_encode($payload_array);
        

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/users/register',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        // La API devuelve un 200 o 201 tanto si el usuario es nuevo como si ya existía.
        // Lo importante es si devuelve una API key.
        if ($http_code >= 200 && $http_code < 300 && !empty($responseData['apiKey'])) {
            // Éxito: se ha recibido una API key
            Configuration::updateValue('AIIMAGEEDIT_API_KEY_BFL', $responseData['apiKey']);
            Configuration::deleteByName('AIIMAGEEDIT_REG_ERROR'); // Limpiar cualquier error de registro previo
            
            $this->ajaxDieJson([
                'success' => true,
                'message' => $responseData['message'] ?? $this->module->l('Registration successful! The page will now reload.')
            ]);
        } else {
            // Fallo: la API devolvió un error
            $errorMessage = $this->l('API Registration Failed. ');
            if (isset($responseData['message'])) {
                $errorMessage .= 'API message: ' . $responseData['message'];
            } elseif(isset($responseData['errors'][0]['msg'])) {
                $errorMessage .= 'API validation error: ' . $responseData['errors'][0]['msg'];
            } else {
                $errorMessage .= $this->module->l('Received HTTP status code: ') . $http_code;
            }
            $this->ajaxDieJson(['success' => false, 'message' => $errorMessage]);
        }
    }

	
public function ajaxProcessGetTopUpUrl()
    {
        $apiKey = AiImageEdit::getBflApiKey();
        if (empty($apiKey)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('API Key is not configured.')]);
        }
        $payload = json_encode(['priceId' => 'price_1Rg7KQRgNtKPUT8PgsBs4SIG']);

        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/payments/create-checkout-session',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'x-api-key: ' . $apiKey,
            ],
        ]);
        
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300 && !empty($responseData['checkout_url'])) {
            $this->ajaxDieJson(['success' => true, 'checkout_url' => $responseData['checkout_url']]);
        } else {
            $errorMessage = $responseData['message'] ?? $this->module->l('Could not get payment link.');
            $this->ajaxDieJson(['success' => false, 'message' => $errorMessage]);
        }
    }

	
	
    public function ajaxProcessGetSubscriptionPortalUrl()
    {
        $apiKey = AiImageEdit::getBflApiKey();
        if (empty($apiKey)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('API Key is not configured.')]);
        }
        
        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/account/stripe-portal',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => ['x-api-key: ' . $apiKey],
        ]);
        
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        
        $responseData = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300 && !empty($responseData['portal_url'])) {
            $this->ajaxDieJson(['success' => true, 'portal_url' => $responseData['portal_url']]);
        } else {
            $errorMessage = $responseData['message'] ?? $this->module->l('Could not retrieve subscription portal URL.');
            $this->ajaxDieJson(['success' => false, 'message' => $errorMessage]);
        }
    }

/**
     * AJAX action to fetch an external image and return its Base64 representation.
     * This acts as a proxy to bypass client-side CORS issues.
     */
    public function ajaxProcessGetImageAsBase64()
    {
        $imageUrl = Tools::getValue('image_url');
        if (!Validate::isAbsoluteUrl($imageUrl)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Invalid URL provided for Base64 conversion.')]);
        }

        // Use Tools::file_get_contents which is a wrapper around file_get_contents with some checks.
        // It might be blocked by allow_url_fopen in php.ini. If so, a cURL fallback is needed.
        $imageData = Tools::file_get_contents($imageUrl);

        if ($imageData === false) {
            // Fallback to cURL if file_get_contents fails (e.g., allow_url_fopen=off)
            $curl = curl_init();
			if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
            curl_setopt($curl, CURLOPT_URL, $imageUrl);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_TIMEOUT, 15);
            $imageData = curl_exec($curl);
            $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            
            if ($imageData === false || $http_code >= 400) {
                 $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to download image from the provided URL. Server might be blocking outbound requests.')]);
            }
        }
        
        // It's good practice to quickly validate if what we downloaded is an image
        if (!@getimagesizefromstring($imageData)) {
             $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('The downloaded content is not a valid image.')]);
        }


        $base64String = base64_encode($imageData);

        $this->ajaxDieJson(['success' => true, 'base64' => $base64String]);
    }
	
public function ajaxProcessTestMyAjax()
{
    $this->ajaxDieJson(['success' => true, 'message' => 'Test AJAX endpoint reached!']);
}	
	
    // --- BFL.AI FLUX KONTEXT PRO API ACTIONS ---
    public function ajaxProcessCreateAiRequest()
    {
        $bflApiKey = AiImageEdit::getBflApiKey();
        if (empty($bflApiKey)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('API Key is not configured.')]);
        }

        $prompt = Tools::getValue('prompt');
        $input_image_base64 = Tools::getValue('input_image_base64'); // Expecting without data:image/... prefix
        $moderation_level = (int)Tools::getValue('moderation_level', 2);
        $output_format = Tools::strtolower(Tools::getValue('output_format', 'jpeg'));
        $seed = Tools::getValue('seed'); // Can be empty string or number
        $prompt_upsampling = Tools::getValue('prompt_upsampling')==1;
		$model_type = Tools::getValue('model', 'Pro');

        if (empty($prompt) || empty($input_image_base64)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Prompt and base input image are required.')]);
        }

        $payload = [
            'prompt' => $prompt,
            'input_image' => $input_image_base64,
            'moderation_level' => $moderation_level,
            'output_format' => $output_format,
            'prompt_upsampling' => $prompt_upsampling,
        ];
        if (!empty($seed) && is_numeric($seed)) {
            $payload['seed'] = (int)$seed;
        } elseif (!empty($seed)) { // If seed is provided but not numeric, BFL might handle it or error.
            $payload['seed'] = $seed; // Or decide to error out if seed must be int
        }

		$endpoint_url = '';
		if (strtolower($model_type) === 'ultra') {
			$endpoint_url = 'https://services.aisyncrasy.com/api/v1/blackforest/generate-ultra';
		} else {
			$endpoint_url = 'https://services.aisyncrasy.com/api/v1/blackforest/generate-pro';
		}
		
        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => $endpoint_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30, // Timeout for the request itself
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'accept: application/json',
                'x-api-key: ' . $bflApiKey,
                'Content-Type: application/json',
            ],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        curl_close($curl);

        if ($curl_error) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('cURL Error creating BFL request: ') . $curl_error]);
        }

        $responseData = json_decode($response, true);

        if ($http_code >= 200 && $http_code < 300 && isset($responseData['id'])) {
            $this->ajaxDieJson(['success' => true, 'request_id' => $responseData['id'], 'message' => $this->module->l('Edit request submitted.')]);
        } else {
            $errorMsg = $this->module->l('Error from edit (Create Request): ');
            if (isset($responseData['detail'])) {
                if (is_array($responseData['detail'])) {
                    foreach ($responseData['detail'] as $det) {
                        $errorMsg .= (isset($det['loc']) ? implode('.', $det['loc']) . ': ' : '') . ($det['msg'] ?? 'Unknown error') . '; ';
                    }
                } else {
                    $errorMsg .= $responseData['detail'];
                }
            } elseif ($response) {
                 $errorMsg .= $response;
            } else {
                $errorMsg .= $this->module->l('Unknown API error. HTTP Code: ') . $http_code;
            }
            
            $this->ajaxDieJson(['success' => false, 'message' => $errorMsg]);
        }
    }

    public function ajaxProcessPollAiResult()
    {
        $bflApiKey = AiImageEdit::getBflApiKey();
        if (empty($bflApiKey)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('AI API Key is not configured.')]);
        }

        $requestId = Tools::getValue('request_id');
        if (empty($requestId)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Edit Request ID is missing for polling.')]);
        }

        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/blackforest/results/' . urlencode($requestId),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 15, // Shorter timeout for polling
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => [
                'accept: application/json',
                'x-api-key: ' . $bflApiKey,
            ],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        curl_close($curl);

        if ($curl_error) {
            $this->ajaxDieJson(['success' => false, 'status' => 'Error', 'message' => $this->module->l('Error polling Edit result: ') . $curl_error]);
        }

        $responseData = json_decode($response, true);

		if ($http_code === 404 && isset($responseData['status']) && $responseData['status'] === 'Task not found') {
			$this->ajaxDieJson([
				'success' => false,
				'status' => 'Task not found',
				'message' => $this->module->l('Task not found on Edit, will retry...')
			]);
		}
        if ($http_code >= 200 && $http_code < 300 && isset($responseData['status'])) {
            $returnPayload = [
                'success' => true, // Indicates AJAX call success, not necessarily generation completion
                'status' => $responseData['status'],
                'progress' => $responseData['progress'] ?? null,
                'image_url' => null,
                'message' => $responseData['error'] ?? ($responseData['message'] ?? null)
            ];
            if ($responseData['status'] === 'Ready' && isset($responseData['result']['sample'])) {
                $returnPayload['image_url'] = $responseData['result']['sample'];
            } elseif ($responseData['status'] !== 'Processing' && $responseData['status'] !== 'Queued' && $responseData['status'] !== 'Pending' && $responseData['status'] !== 'Ready') {
                // An error status from BFL
                $returnPayload['success'] = false; // Set overall success to false if BFL reports an error status
                $errorMsg = $this->module->l('Error status from Edit (Poll): ') . $responseData['status'];
                 if (isset($responseData['error'])) {
                    $errorMsg .= ' - ' . $responseData['error'];
                } elseif (isset($responseData['detail'])) {
                    $errorMsg .= ' - ' . (is_array($responseData['detail']) ? json_encode($responseData['detail']) : $responseData['detail']);
                }
                $returnPayload['message'] = $errorMsg;
            }
            $this->ajaxDieJson($returnPayload);
        } else {
            $errorMsg = $this->module->l('Error from Edit (Poll Result). HTTP Code: ') . $http_code;
            if ($response) $errorMsg .= ' Response: '.$response;
            $this->ajaxDieJson(['success' => false, 'status' => 'Error', 'message' => $errorMsg]);
        }
    }

    // --- REPLICATE API ACTIONS ---

    public function ajaxProcessCreateUpscaleRequest()
    {
		$bflApiKey = AiImageEdit::getBflApiKey();

        $image_url_to_upscale = Tools::getValue('image_url');
        $scale_factor = (float)Tools::getValue('scale_factor', 2.0);
        $face_enhance = Tools::getValue('face_enhance')==1;

        if (!Validate::isAbsoluteUrl($image_url_to_upscale)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Valid image URL is required for upscaling.')]);
        }
        if ($scale_factor <= 0 || $scale_factor > 10) { // Replicate might have its own limits
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Invalid scale factor. Must be between 0.1 and 10 (example range).')]);
        }

        $payload = [
                'image' => $image_url_to_upscale,
                'scale' => $scale_factor,
                'face_enhance' => $face_enhance 
        ];
		
		
 
        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/replicate/run-esrgan',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'x-api-key: ' . $bflApiKey, // Replicate uses "Token" prefix
            ],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        curl_close($curl);

        if ($curl_error) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('cURL Error creating Replicate request: ') . $curl_error]);
        }

        $responseData = json_decode($response, true);

        if ($http_code >= 200 && $http_code < 300 && isset($responseData['id'])) {
            $this->ajaxDieJson(['success' => true, 'request_id' => $responseData['id'], 'message' => $this->module->l('Replicate upscale request submitted.')]);
        } else {
            $errorMsg = $this->module->l('Error from Scale (Create Prediction): ');
            $errorMsg .= isset($responseData['detail']) ? $responseData['detail'] : ($response ? $response : $this->module->l('Unknown API error. HTTP Code: ') . $http_code);
            
            $this->ajaxDieJson(['success' => false, 'message' => $errorMsg]);
        }
    }

    public function ajaxProcessPollUpscaleResult()
    {
        $bflApiKey = AiImageEdit::getBflApiKey();
        $requestId = Tools::getValue('request_id');

        if (empty($requestId)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Scale Request ID is missing for polling.')]);
        }

        $curl = curl_init();
		if ($curl === false) {
    		$this->ajaxDieJson(['success' => false, 'message' => 'Failed to initialize cURL']);
		}
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/replicate/predictions/' . urlencode($requestId),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => [
                'x-api-key: ' . $bflApiKey,
            ],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        curl_close($curl);

        if ($curl_error) {
            $this->ajaxDieJson(['success' => false, 'status' => 'error', 'message' => $this->module->l('Error polling Scale result: ') . $curl_error]);
        }

        $responseData = json_decode($response, true);

        if ($http_code >= 200 && $http_code < 300 && isset($responseData['status'])) {
            $returnPayload = [
                'success' => true,
                'status' => $responseData['status'], // e.g., "starting", "processing", "succeeded", "failed", "canceled"
                'image_url' => null,
                'message' => $responseData['error'] ?? null, // Replicate puts error string in 'error' field
                'logs' => $responseData['logs'] ?? ''
            ];
            if ($responseData['status'] === 'succeeded' && !empty($responseData['output'])) {
                // Output can be an array or a string. For Real-ESRGAN it's usually a string URL.
                $returnPayload['image_url'] = is_array($responseData['output']) ? $responseData['output'][0] : $responseData['output'];
            } elseif ($responseData['status'] === 'failed' || $responseData['status'] === 'canceled') {
                $returnPayload['success'] = false; // Overall success false if Replicate task failed
                $errorMsg = $this->module->l('Replicate task status: ') . $responseData['status'];
                if (!empty($responseData['error'])) $errorMsg .= ' - ' . $responseData['error'];
                $returnPayload['message'] = $errorMsg;
            }
            $this->ajaxDieJson($returnPayload);
        } else {
            $errorMsg = $this->module->l('Error from Scale (Poll Prediction). HTTP Code: ') . $http_code;
            if ($response) $errorMsg .= ' Response: '. $response;
            $this->ajaxDieJson(['success' => false, 'status' => 'error', 'message' => $errorMsg]);
        }
    }

    // --- AI PARAMETER CONFIGURATION ACTIONS ---
    // (These interact with your custom aiimageedit_configs table)

    public function ajaxProcessSaveAiConfig()
    {
	     $id_shop = (int)$this->context->shop->id;
        $config_name = Tools::getValue('config_name');
        if (empty($config_name)) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Configuration name is required.')]);
        }

        $params = [
            'id_shop' => $id_shop,
            'name' => $config_name,
            'prompt' => Tools::getValue('prompt', ''), // Allow empty prompt
            'num_images' => (int)Tools::getValue('num_images', 1),
            'moderation_level' => (int)Tools::getValue('moderation_level', 2),
            'seed' => Tools::getValue('seed', null), // Store as string or null
            'output_format' => Tools::getValue('output_format', 'jpeg'),
            'prompt_upsampling' => (int)Tools::getValue('prompt_upsampling', 1),
			'model' => Tools::getValue('model', 'Pro'), 
            'upscale_target_specific' => (int)Tools::getValue('upscale_target_specific', 1),
            'upscale_factor' => (float)Tools::getValue('upscale_factor', 1.5),
            'upscale_face_enhance' =>  (int)Tools::getValue('upscale_face_enhance', 1),
            'date_add' => date('Y-m-d H:i:s'),
            'date_upd' => date('Y-m-d H:i:s'),
        ];
        if (empty($params['seed'])) { // Handle empty string seed as NULL for DB
            $params['seed'] = null;
        }


        // Check if config with this name for this shop already exists
        $existing_id = Db::getInstance()->getValue('
            SELECT `id_aiimageedit_config`
            FROM `'._DB_PREFIX_.AiImageEdit::CONFIG_TABLE_NAME.'`
            WHERE `id_shop` = '.$id_shop.' AND `name` = "'.pSQL($config_name).'"
        ');

        if ($existing_id) {
            unset($params['date_add']); // Don't update date_add
            $result = Db::getInstance()->update(AiImageEdit::CONFIG_TABLE_NAME, $params, 'id_aiimageedit_config = '.(int)$existing_id);
            $action_message = $this->module->l('Configuration updated: ') . $config_name;
            $new_config_id = (int)$existing_id;
        } else {
            $result = Db::getInstance()->insert(AiImageEdit::CONFIG_TABLE_NAME, $params);
            $action_message = $this->module->l('Configuration saved: ') . $config_name;
            $new_config_id = (int)Db::getInstance()->Insert_ID();
        }

        if ($result) {
            $this->ajaxDieJson(['success' => true, 'message' => $action_message, 'config_id' => $new_config_id, 'config_name' => $config_name]);
        } else {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to save configuration to database.') . Db::getInstance()->getMsgError()]);
        }
    }

    public function ajaxProcessLoadAiConfig()
    {
        $id_config = (int)Tools::getValue('id_config');
        $id_shop = (int)$this->context->shop->id;

        if (!$id_config) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Invalid configuration ID.')]);
        }

        $config = Db::getInstance()->getRow('
            SELECT * FROM `'._DB_PREFIX_.AiImageEdit::CONFIG_TABLE_NAME.'`
            WHERE `id_aiimageedit_config` = '.$id_config.' AND `id_shop` = '.$id_shop
        );

        if ($config) {
            // Cast boolean-like values back for JS if needed (JS handles 0/1 fine for checkboxes)
            $config['prompt_upsampling'] = (bool)$config['prompt_upsampling'];
            $config['upscale_target_specific'] = (bool)$config['upscale_target_specific'];
            $config['upscale_face_enhance'] = (bool)$config['upscale_face_enhance'];
            $config['num_images'] = (int)$config['num_images'];
            $config['moderation_level'] = (int)$config['moderation_level'];
            $config['seed'] = $config['seed'] === null ? '' : $config['seed']; // JS expects empty string for empty input

            $this->ajaxDieJson(['success' => true, 'config' => $config]);
        } else {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Configuration not found.')]);
        }
    }

    public function ajaxProcessDeleteAiConfig()
    {
        $id_config = (int)Tools::getValue('id_config');
        $id_shop = (int)$this->context->shop->id;

        if (!$id_config) {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Invalid configuration ID.')]);
        }

        $result = Db::getInstance()->delete(
            AiImageEdit::CONFIG_TABLE_NAME,
            '`id_aiimageedit_config` = '.$id_config.' AND `id_shop` = '.$id_shop
        );

        if ($result) {
            $this->ajaxDieJson(['success' => true, 'message' => $this->module->l('Configuration deleted.')]);
        } else {
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to delete configuration.')]);
        }
    }
    
    // Static version for the hook in module file
    public static function getSavedConfigsStatic()
    {
        $id_shop = (int)Context::getContext()->shop->id;
        $sql = new DbQuery();
        $sql->select('id_aiimageedit_config, name');
        $sql->from(AiImageEdit::CONFIG_TABLE_NAME);
        $sql->where('id_shop = ' . $id_shop);
        $sql->orderBy('name ASC');
        try {
            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);
        } catch (PrestaShopDatabaseException $e) {
            return [];
        }
        return $results ? $results : [];
    }


    public function ajaxProcessGetAiConfigs() // Called by JS to refresh dropdown
    {
        $this->ajaxDieJson(['success' => true, 'configs' => self::getSavedConfigsStatic()]);
    }

// In AdminAiImageEditController.php
public function ajaxProcessSaveDefaultAiParams()
{
 
	$params = [
        'prompt' => Tools::getValue('prompt', ''), // Allow empty to clear default
        'num_images' => (int)Tools::getValue('num_images', 1),
        'moderation_level' => (int)Tools::getValue('moderation_level', 2),
        'seed' => Tools::getValue('seed', ''), // Store as string
        'output_format' => Tools::getValue('output_format', 'jpeg'),
		'model' => Tools::getValue('model', 'Pro'),
        'upscale_factor' => (float)Tools::getValue('upscale_factor', 2), 
		 // --- CONVERSIÓN CORRECTA DE 1/0 A BOOLEANO ---
        'prompt_upsampling' => (bool)(int)Tools::getValue('prompt_upsampling', 1),
        'upscale_target_specific' => (bool)(int)Tools::getValue('upscale_target_specific', 1),
        'upscale_face_enhance' => (bool)(int)Tools::getValue('upscale_face_enhance', 1),
    ];

    // id_shop_group = null, id_shop = current shop context or null for all shops
    // The false for html is important for json_encode not to double-encode html entities
    if (Configuration::updateValue(AiImageEdit::DEFAULT_CONFIG_NAME, json_encode($params), true, null, (int)$this->context->shop->id)) {
        $this->ajaxDieJson(['success' => true, 'message' => $this->module->l('Default parameters saved successfully for this shop.')]);
    } else {
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to save default parameters.')]);
    }
}

    // --- OLDER COPY IMAGE ACTION (kept for reference, can be removed if not used) ---
    public function ajaxProcessCopyImage()
    {
        // ... Your existing ajaxProcessCopyImage logic ...
        // This is probably not needed anymore given the new workflow.
        // If you remove it, also remove the "Copy Image" button from the TPL.
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Copy function is disabled in this version.')]);
    }
	
    // ... your ajaxProcessCopyImage and ajaxProcessSaveAiGeneratedImage methods ...
    // Make sure they are also correctly using $this->ajaxDie(json_encode([...])) for all exit paths.
    // The ajaxProcessCopyImage and ajaxProcessSaveAiGeneratedImage methods you provided earlier looked okay in this regard.

    public function ajaxProcessSaveAiGeneratedImage()
{
    // --- 1. Get Initial Data & Determine Target Format Based on Global PS Config ---
    $legend_text = Tools::getValue('legend', $this->module->l('generated by aisyncrasy.com'));
    $id_product = (int)Tools::getValue('id_product');
    $generated_image_url = Tools::getValue('generated_image_url');
    $legend_base = Tools::getValue('legend_base', $this->module->l('AI Modified Image'));

    // NEW: Server-side format determination. We ignore the 'output_format' from the client.
    $ps_image_format = Configuration::get('PS_IMAGE_FORMAT');
    // Rule: Convert to JPG unless the shop is configured to use PNG *exclusively*.
    if ($ps_image_format === 'webp') {
		$final_extension = 'webp';
 	} elseif ($ps_image_format === 'png')
	{
        $final_extension = 'png';
    } else {
        $final_extension = 'jpg';
    }



    if (!$id_product || !Validate::isAbsoluteUrl($generated_image_url)) {
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Missing product ID or invalid URL.')]);
    }
    
    // --- 2. Download Image to a Temporary File ---
    $temp_image_path = tempnam(_PS_TMP_IMG_DIR_, 'ai_gen_');
    if (!Tools::copy($generated_image_url, $temp_image_path)) {
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to download the generated image.')]);
    }
 
    if (!ImageManager::isRealImage($temp_image_path)) {
        @unlink($temp_image_path);
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Downloaded file is not a valid image.')]);
    }
    
    // --- 3. Convert Image if Necessary (Based on our new rule) ---
    $source_image_path = $temp_image_path; // The file we will copy to the final destination
    $source_mime_type = mime_content_type($source_image_path);
    
    $needs_conversion = false;
    if ($final_extension === 'jpg' && $source_mime_type !== 'image/jpeg') {
        $needs_conversion = true;
    } elseif ($final_extension === 'png' && $source_mime_type !== 'image/png') {
        $needs_conversion = true;
    }

// --- REVISED: Conversion logic with WEBP support ---
        $source_mime_type_map = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/webp' => 'webp',
            'image/gif' => 'gif',
        ];
        $source_extension = $source_mime_type_map[$source_mime_type] ?? null;

        if ($source_extension !== $final_extension) {
           
            $converted_image_path = _PS_TMP_IMG_DIR_ . 'ai_converted_' . uniqid() . '.' . $final_extension;
            $image_resource = null;

            // Create resource from source
            if ($source_extension === 'png') $image_resource = @imagecreatefrompng($source_image_path);
            elseif ($source_extension === 'gif') $image_resource = @imagecreatefromgif($source_image_path);
            elseif ($source_extension === 'webp') $image_resource = @imagecreatefromwebp($source_image_path);
            elseif ($source_extension === 'jpg') $image_resource = @imagecreatefromjpeg($source_image_path);

            if (!$image_resource) {
                @unlink($source_image_path);
 
                $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Could not process the downloaded image for conversion.')]);
            }
            
            $save_success = false;
            // Save resource to target format
            if ($final_extension === 'jpg') {
                $jpeg_quality = (int)Configuration::get('PS_JPEG_QUALITY', 90);
                $save_success = @imagejpeg($image_resource, $converted_image_path, $jpeg_quality);
            } elseif ($final_extension === 'png') {
                $png_compression = (int)Configuration::get('PS_PNG_QUALITY', 7);
                // For PNG conversion, preserve transparency
                imagepalettetotruecolor($image_resource);
                imagealphablending($image_resource, true);
                imagesavealpha($image_resource, true);
                $save_success = @imagepng($image_resource, $converted_image_path, $png_compression);
            } elseif ($final_extension === 'webp') {
                $webp_quality = (int)Configuration::get('PS_WEBP_QUALITY', 90);
                // For WEBP conversion, preserve transparency
                imagepalettetotruecolor($image_resource);
                imagealphablending($image_resource, true);
                imagesavealpha($image_resource, true);
                $save_success = @imagewebp($image_resource, $converted_image_path, $webp_quality);
            }

            imagedestroy($image_resource);

            if ($save_success) {
                @unlink($source_image_path); // Delete original temp file
                $source_image_path = $converted_image_path; // The new source is the converted file
            } else {
                @unlink($source_image_path);
                $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to save the image in the target format.')]);
            }
        }
    
    // --- 4. Create PrestaShop Image Object and Save File ---
    $new_image = new Image();
    $new_image->id_product = $id_product;
    $new_image->position = Image::getHighestPosition($id_product) + 1;
    $new_image->cover = null;
    foreach (Language::getLanguages(true) as $lang) {
        $new_image->legend[$lang['id_lang']] = $legend_text;
    }

    if (!$new_image->add()) {
        @unlink($source_image_path);
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to create new image database record.')]);
    }
 
    // Step 5: *** THE CRITICAL FIX: MANUALLY CREATE THE DIRECTORY ***
    // This is the absolute requirement to fix your "No such file or directory" error.
    
    // getImgPath() returns the path *including* the filename base, e.g., '1/3/4/8/2/13482'
    // We use dirname() to get just the directory part of that path.
    $destination_folder_path = _PS_PROD_IMG_DIR_ . dirname($new_image->getImgPath());
    
    // Check if the directory exists. If not, create it recursively.
    if (!is_dir($destination_folder_path)) {
        // The 'true' parameter makes mkdir recursive, creating all parent directories as needed.
        if (!@mkdir($destination_folder_path, 0775, true)) {
            // If directory creation fails, it's a critical server permission error.
            // We must clean up the database record we just made.
            $new_image->delete();
            @unlink($source_image_path);
            $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Fatal Error: Failed to create product image directory. Please check server permissions on the `img/p/` folder.')]);
        }
    }
    // --- END OF CRITICAL FIX ---		
		
    $new_final_file_path_base = _PS_PROD_IMG_DIR_ . $new_image->getImgPath();
    $new_final_file_path = $new_final_file_path_base . '.' . $final_extension;
    
    if (!copy($source_image_path, $new_final_file_path)) {
        $new_image->delete();
        @unlink($source_image_path);
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to copy image to product directory.')]);
    }

    $new_image->image_format = $final_extension;
    if (!$new_image->update()) {
        @unlink($source_image_path); @unlink($new_final_file_path);
        $new_image->delete();
        $this->ajaxDieJson(['success' => false, 'message' => $this->module->l('Failed to update image format in database.')]);
    }
    
    @unlink($source_image_path);

    // --- 5. Thumbnail Generation ---
// --- Step 7: Thumbnail Generation (Manual Loop for Compatibility) ---
        
        $image_types = ImageType::getImagesTypes('products');
        $errors = [];
        
        // Temporarily set a custom error handler to catch warnings from GD/ImageMagick
        set_error_handler(function ($severity, $message, $file, $line) use (&$errors) {
            $errors[] = "PHP Warning/Notice: " . $message;
        });

        foreach ($image_types as $image_type) {
            $thumb_destination = $new_final_file_path_base . '-' . $image_type['name'] . '.' . $final_extension;
            
            if (!ImageManager::resize(
                $new_final_file_path, // The full source image path
                $thumb_destination,
                (int)$image_type['width'],
                (int)$image_type['height'],
                $final_extension
            )) {
                $errors[] = sprintf($this->module->l('Resize function returned false for type %s.'), $image_type['name']);
            }
        }
        
        // Restore the default error handler
        restore_error_handler();

        if (!empty($errors)) {
            // If thumbnails failed, the image might be "invisible". This is a critical failure.
            $error_message_string = $this->module->l('Image record created, but thumbnail generation failed. The image may not be visible. Errors: ') . implode('; ', $errors);
            
            // It's often better to not return success if thumbnails fail, as the image will appear broken.
            $this->ajaxDieJson(['success' => false, 'message' => $error_message_string]);
        }
    // --- 6. Finalization & Cache Clearing ---
    $product = new Product($id_product);
    if (Validate::isLoadedObject($product)) {
        $product->clearCache();
    }
    if (method_exists('Product', 'cleanPositions')) {
        Product::cleanPositions($id_product);
    }
    Hook::exec('actionProductUpdate', ['id_product' => $id_product, 'product' => $product]);

  		
	// REFRESCO DE LA GALERIA Volvemos a instanciar el producto para obtener el link_rewrite correcto
    $product_for_link = new Product($id_product, false, $this->context->language->id);
    $link_rewrite = 'product'; // Fallback
    if (Validate::isLoadedObject($product_for_link) && !empty($product_for_link->link_rewrite[$this->context->language->id])) {
        $link_rewrite = $product_for_link->link_rewrite[$this->context->language->id];
    }

    // Usamos un tipo de miniatura estándar para la galería superior.
    // 'cart_default' o 'small_default' suelen ser buenas opciones.
    $thumbnail_type_name = ImageType::getFormattedName('small');

    // Obtenemos las dimensiones de la imagen guardada
    $image_path = _PS_PROD_IMG_DIR_ . $new_image->getImgPath() . '.' . $new_image->image_format;
    $dims = file_exists($image_path) ? @getimagesize($image_path) : null;
    
    // Construimos el objeto con los datos de la nueva imagen
    $new_image_data_for_js = [
        'id_image' => $new_image->id,
        'url' => $this->context->link->getImageLink($link_rewrite, $new_image->id, $thumbnail_type_name),
        'original_url' => $this->context->link->getImageLink($link_rewrite, $new_image->id),
        'legend' => $new_image->legend[$this->context->language->id] ?? '',
        'width' => $dims[0] ?? null,
        'height' => $dims[1] ?? null,
    ];	
    $this->ajaxDieJson(['success' => true, 'message' => $this->module->l('AI Image saved successfully!'), 'new_image_data' => $new_image_data_for_js]);
}

}