<?php
/**
 * 2007-2023 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    AISyncrasy
 * @copyright 2025 Modasyncrasy S.L.
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

// Ensure AdminAiImageEditController can be found for static calls if needed from hooks
// (though direct static calls from hook to controller are not typical; data usually passed via Smarty)
// If your controller has static helper methods used by the module class, ensure it's loadable.
// require_once _PS_MODULE_DIR_ . 'aiimageedit/controllers/admin/AdminAiImageEditController.php';


class AiImageEdit extends Module
{
    const CONFIG_API_KEY_BFL = 'AIIMAGEEDIT_API_KEY_BFL';
	const CONFIG_REGISTRATION_ERROR = 'AIIMAGEEDIT_REG_ERROR'; // Para guardar errores de registro
    const CONFIG_ACCOUNT_ID = 'AIIMAGEEDIT_ACCOUNT_ID'; // Para guardar el ID de usuario de la AP
    // const CONFIG_API_KEY_REPLICATE = 'AIIMAGEEDIT_API_KEY_REPLICATE'; // For future config
    const CONFIG_TABLE_NAME = 'aiimageedit_configs'; // Custom table for saved parameter sets
    const DEFAULT_CONFIG_NAME = 'AIIMAGEEDIT_DEFAULT_PARAMS'; // Key for ps_configuration

    public function __construct()
    {
        $this->name = 'aiimageedit';
        $this->tab = 'administration'; // Or 'content_management', 'products'
        $this->version = '1.2.0'; // Increment for new features
        $this->author = 'AISyncrasy';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '1.7.0', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('AISYNCRASY AI Image Edit and Upscale');
        $this->description = $this->l('Advanced AI image editing, generation and upscaling for product images.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall? This will remove API key configurations and custom parameter sets.');

        if (!Configuration::get(self::CONFIG_API_KEY_BFL)) {
            $this->warning = $this->l('API Key is not yet configured. Image generation will not work.');
        }
    }

	
 private function registerAndFetchApiKey()
    {
        $domain_full = Tools::getShopDomainSSL(true, true);
        $domain = preg_replace('#^https?://#', '', rtrim($domain_full, '/'));
        $admin_email = isset($this->context->employee->email) && !empty($this->context->employee->email) 
            ? $this->context->employee->email 
            : Configuration::get('PS_SHOP_EMAIL');

        // --- CORRECCIÓN DE SINTAXIS ---
        $payload_array = [
            'origin' => 'prestashop',
            'domain' => $domain,
            'email' => $admin_email,
        ]; // <-- Se reemplazó el ')' por un ';'

        $payload = json_encode($payload_array);
        
        PrestaShopLogger::addLog('AIImageEdit: Attempting auto-registration with payload: ' . print_r($payload_array, true), 1);

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/users/register',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        $responseData = json_decode($response, true);

        if ($http_code >= 200 && $http_code < 300 && !empty($responseData['apiKey'])) {
            Configuration::updateValue(self::CONFIG_API_KEY_BFL, $responseData['apiKey']);
            Configuration::deleteByName(self::CONFIG_REGISTRATION_ERROR);
            return true;
        } else {
            $errorMessage = $this->l('Automatic registration failed. ');
            if (isset($responseData['message'])) {
                $errorMessage .= 'API message: ' . $responseData['message'];
            } elseif(isset($responseData['errors'][0]['msg'])) {
                $errorMessage .= 'API validation error: ' . $responseData['errors'][0]['msg'];
            } else {
                $errorMessage .= $this->l('Received HTTP status code: ') . $http_code;
            }
            Configuration::updateValue(self::CONFIG_REGISTRATION_ERROR, $errorMessage);
            PrestaShopLogger::addLog('AIImageEdit: Registration failed. ' . $errorMessage, 3);
            return false;
        }
    }

    /**
     * Defines the default parameters for AI operations.
     * @return array
     */
    private function getDefaultParams() {
        return [
            'prompt' => '',
            'num_images' => 1, // How many BFL requests to make
            'moderation_level' => 2,
            'seed' => '',
            'output_format' => 'jpeg', // For BFL
            'prompt_upsampling' => true,
			'model' => 'Pro',
            'upscale_target_specific' => true, // true = "Scale to [WxH]", false = "Scale xFactor"
            'upscale_factor' => 2.0, // Default factor if not specific (e.g. for "Scale x2")
            'upscale_face_enhance' => true
        ];
    }

       public function install()
    {
        // --- LÓGICA DE INSTALACIÓN MEJORADA ---
        if (!parent::install() ||
		    !$this->installController() ||
            !$this->registerHook('displayAdminProductsExtra') ||
            !$this->installDb()) {
            return false;
        }

        // Inicializar la configuración
        Configuration::updateValue(self::CONFIG_API_KEY_BFL, '');
        Configuration::updateValue(self::DEFAULT_CONFIG_NAME, json_encode($this->getDefaultParams()));

        // Intentar el registro automático, pero no hacer que la instalación falle si no funciona.
        // El usuario podrá obtener una clave manualmente desde la página de configuración.
        $this->registerAndFetchApiKey();
        
        return true;
    }

private function installController()
{
    // Register the controller as a hidden tab (no menu entry)
    $tab = new Tab();
    $tab->class_name = 'AdminAiImageEdit';
    $tab->module = $this->name;
    $tab->id_parent = -1; // -1 = hidden tab
    $tab->active = 1;
    
    foreach (Language::getLanguages(false) as $lang) {
        $tab->name[(int)$lang['id_lang']] = 'AI Image Edit Controller';
    }
    
    return $tab->add();
}
	
private function uninstallController()
{
    $id_tab = (int)Tab::getIdFromClassName('AdminAiImageEdit');
    if ($id_tab) {
        $tab = new Tab($id_tab);
        return $tab->delete();
    }
    return true;
}

public function uninstall()
{
    PrestaShopLogger::addLog('AIImageEdit: Starting uninstallation...', 1);
    $success = true;

    // Eliminar configuraciones
    if (!Configuration::deleteByName(self::CONFIG_API_KEY_BFL)) {
        PrestaShopLogger::addLog('AIImageEdit: Failed to delete BFL API Key configuration.', 2);
    }
    if (!Configuration::deleteByName(self::CONFIG_REGISTRATION_ERROR)) {
        PrestaShopLogger::addLog('AIImageEdit: Failed to delete registration error configuration.', 2);
    }
    if (!Configuration::deleteByName(self::CONFIG_ACCOUNT_ID)) {
        PrestaShopLogger::addLog('AIImageEdit: Failed to delete account ID configuration.', 2);
    }
    if (!Configuration::deleteByName(self::DEFAULT_CONFIG_NAME)) {
        PrestaShopLogger::addLog('AIImageEdit: Failed to delete default parameters configuration.', 2);
    }

    // Desinstalar controlador
    if (!$this->uninstallController()) {
        PrestaShopLogger::addLog('AIImageEdit: uninstallController() returned false.', 3);
        $this->_errors[] = $this->l('Failed to uninstall controller.');
        $success = false;
    }

    // Desinstalar base de datos
    if (!$this->uninstallDb()) {
        PrestaShopLogger::addLog('AIImageEdit: uninstallDb() returned false.', 3);
        $this->_errors[] = $this->l('Failed to uninstall database table.');
        $success = false;
    }

    // Llamar al método padre una sola vez
    if (!parent::uninstall()) {
        PrestaShopLogger::addLog('AIImageEdit: parent::uninstall() failed.', 3);
        $success = false;
    }
    
    PrestaShopLogger::addLog('AIImageEdit: Uninstallation completed (Success: ' . ($success ? 'true' : 'false') . ').', 1);
    return $success;
}

    public function installDb()
    {
        $tableName = _DB_PREFIX_ . self::CONFIG_TABLE_NAME;
        $sql = "CREATE TABLE IF NOT EXISTS `{$tableName}` (
            `id_aiimageedit_config` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
            `id_shop` INT(11) UNSIGNED NOT NULL DEFAULT '1',
            `name` VARCHAR(255) NOT NULL,
            `prompt` TEXT NULL,
            `num_images` TINYINT(2) UNSIGNED DEFAULT 1,
            `moderation_level` TINYINT(1) UNSIGNED DEFAULT 2,
            `seed` VARCHAR(255) NULL,
			`model` VARCHAR(50) DEFAULT 'Pro',
            `output_format` VARCHAR(10) DEFAULT 'jpeg',
            `prompt_upsampling` TINYINT(1) UNSIGNED DEFAULT 1,
            `upscale_target_specific` TINYINT(1) UNSIGNED DEFAULT 1, /* 1 for specific, 0 for factor */
            `upscale_factor` DECIMAL(5,2) DEFAULT 2.00,
            `upscale_face_enhance` TINYINT(1) UNSIGNED DEFAULT 1,
            `date_add` DATETIME NOT NULL,
            `date_upd` DATETIME NOT NULL,
            PRIMARY KEY (`id_aiimageedit_config`),
            KEY `id_shop_name` (`id_shop`, `name`(191)) /* Max key length for utf8mb4 */
        ) ENGINE=" . _MYSQL_ENGINE_ . " DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

        PrestaShopLogger::addLog('AIImageEdit installDb SQL: ' . preg_replace('/\s+/', ' ', $sql), 1);
        $result = Db::getInstance()->execute($sql);
        if (!$result) {
            $dbError = Db::getInstance()->getMsgError();
            PrestaShopLogger::addLog('AIImageEdit: SQL execution failed for table creation. Error: ' . $dbError, 3);
            $this->_errors[] = $this->l('DB Table Creation Error: ') . $dbError;
        } else {
            PrestaShopLogger::addLog('AIImageEdit: Custom table created (or already existed).', 1);
        }
        return $result;
    }

    public function uninstallDb()
    {
        $tableName = _DB_PREFIX_ . self::CONFIG_TABLE_NAME;
        $sql = "DROP TABLE IF EXISTS `{$tableName}`";
        PrestaShopLogger::addLog('AIImageEdit uninstallDb SQL: ' . $sql, 1);
        $result = Db::getInstance()->execute($sql);
         if (!$result) {
            $dbError = Db::getInstance()->getMsgError();
            PrestaShopLogger::addLog('AIImageEdit: SQL execution failed for table drop. Error: ' . $dbError, 3);
        } else {
            PrestaShopLogger::addLog('AIImageEdit: Custom table dropped.', 1);
        }
        return $result;
    }

	
private function getAccountDetails($apiKey)
{
    if (empty($apiKey)) {
        return ['error' => $this->l('API Key is missing.')];
    }

    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => 'https://services.aisyncrasy.com/api/v1/account/details',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'GET',
        CURLOPT_HTTPHEADER => ['x-api-key: ' . $apiKey],
    ]);

    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($curl);
    curl_close($curl);
    
    // Verificar errores de cURL
    if ($curl_error) {
        PrestaShopLogger::addLog('AIImageEdit: cURL error in getAccountDetails: ' . $curl_error, 3);
        return ['error' => $this->l('Connection error occurred.')];
    }
    
    // Verificar si la respuesta es válida
    if ($response === false) {
        return ['error' => $this->l('No response received from API.')];
    }
    
    $responseData = json_decode($response, true);
    
    // Verificar si el JSON es válido
    if (json_last_error() !== JSON_ERROR_NONE) {
        PrestaShopLogger::addLog('AIImageEdit: Invalid JSON response in getAccountDetails: ' . json_last_error_msg(), 3);
        return ['error' => $this->l('Invalid response format.')];
    }
    
    if ($http_code >= 200 && $http_code < 300 && isset($responseData['id'])) {
        Configuration::updateValue(self::CONFIG_ACCOUNT_ID, $responseData['id']);
        return $responseData;
    } else {
        $errorMsg = isset($responseData['message']) ? $responseData['message'] : 'HTTP ' . $http_code;
        PrestaShopLogger::addLog('AIImageEdit: API error in getAccountDetails: ' . $errorMsg, 3);
        return ['error' => $this->l('Could not retrieve account details. Please check your API Key. Error: ') . $errorMsg];
    }
}

public function getContent()
{
    $output = '';
    
    // IMPORTANTE: Procesar formularios Y redirigir inmediatamente para evitar bucles
    if (Tools::isSubmit('submitRetryRegistration')) {
        $success = $this->registerAndFetchApiKey();
        
        if ($success) {
            // Éxito: limpiar error de registro y redirigir con mensaje de éxito
            Configuration::deleteByName(self::CONFIG_REGISTRATION_ERROR);
            Tools::redirectAdmin($this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&conf=4');
        } else {
            // Error: redirigir con mensaje de error (sin conf=4)
            Tools::redirectAdmin($this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&registration_error=1');
        }
        // Esta línea nunca debería ejecutarse, pero por seguridad:
        exit;
    }

    // Manejar la actualización manual de la API key
    if (Tools::isSubmit('submitUpdateApiKey')) {
        $manualApiKey = Tools::getValue('manual_api_key');
        if (Validate::isString($manualApiKey) && !empty($manualApiKey)) {
            Configuration::updateValue(self::CONFIG_API_KEY_BFL, trim($manualApiKey));
            Configuration::deleteByName(self::CONFIG_REGISTRATION_ERROR); // Limpiar error al poner clave manual
            Tools::redirectAdmin($this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&conf=4');
        } else {
            Tools::redirectAdmin($this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&api_key_error=1');
        }
        exit;
    }

    // Mostrar mensajes basados en parámetros GET (no POST)
    if (Tools::getValue('registration_error') == '1') {
        $output .= $this->displayError($this->l('Registration failed. Please check logs or contact support.'));
    }
    if (Tools::getValue('api_key_error') == '1') {
        $output .= $this->displayError($this->l('The provided API Key is empty or invalid.'));
    }

    // Obtener los datos para mostrar
    $apiKey = Configuration::get(self::CONFIG_API_KEY_BFL);
    $registrationError = Configuration::get(self::CONFIG_REGISTRATION_ERROR);
    
    // Inicializar accountDetails como array vacío
    $accountDetails = [];
    
    // Solo obtener detalles de cuenta si hay API key Y no hay errores de registro
    if (!empty($apiKey) && empty($registrationError)) {
        $accountDetailsResponse = $this->getAccountDetails($apiKey);
        if (is_array($accountDetailsResponse) && !isset($accountDetailsResponse['error'])) {
            $accountDetails = $accountDetailsResponse;
        }
    }

    $this->context->smarty->assign([
        'module_configure_url' => $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name,
        'api_key' => $apiKey ?: '',
        'registration_error' => $registrationError ?: '',
        'account_details' => $accountDetails,
        'controller_url' => $this->context->link->getAdminLink('AdminAiImageEdit'),
        'ajax_token' => Tools::getAdminTokenLite('AdminAiImageEdit'),
        'prestashop_lang_iso' => $this->context->language->iso_code,
        'has_api_key' => !empty($apiKey),
        'has_account_details' => !empty($accountDetails),
    ]);

    return $output . $this->display(__FILE__, 'views/templates/admin/configure.tpl');
}
    public static function getBflApiKey()
    {
        return Configuration::get(self::CONFIG_API_KEY_BFL);
    }

    // Hardcoded as per previous request. Should be configurable.
    public static function getReplicateApiKey() {
        return 'r8_1SOWPQaxiTaX1fRg4HFFtiS7FV7DOvj3GMLqB';
    }

	    public static function getSavedConfigsStatic() // Kept static for direct call
    {
        $id_shop = (int)Context::getContext()->shop->id;
        if (!$id_shop) { // Fallback if context shop is not available (e.g. CLI)
            $id_shop = (int)Configuration::get('PS_SHOP_DEFAULT');
        }

        $sql = new DbQuery();
        $sql->select('id_aiimageedit_config, name');
        $sql->from(self::CONFIG_TABLE_NAME); // Use the constant from this class
        $sql->where('id_shop = ' . $id_shop);
        $sql->orderBy('name ASC');

        try {
            // Use _PS_USE_SQL_SLAVE_ if appropriate for read operations
            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);
        } catch (PrestaShopDatabaseException $e) {
            PrestaShopLogger::addLog(
                'AIImageEdit Module: Error fetching saved configs - ' . $e->getMessage(),
                3,
                null,
                'AiImageEdit' // Specify class name for log
            );
            return []; // Return empty on error
        }
        return $results ? $results : [];
    }
	

 public function hookDisplayAdminProductsExtra($params)
    {
        $id_product = (int)$params['id_product'];
        // Ensure shop context is set for multi-shop scenarios if product is shared
        $product = new Product($id_product, false, $this->context->language->id, $this->context->shop->id);

        if (!Validate::isLoadedObject($product)) {
            return $this->displayError($this->l('Product not found.'));
        }

        $images = Image::getImages($this->context->language->id, $id_product);
        $product_images_data = [];
        $first_image_details = null;

        $image_types = ImageType::getImagesTypes('products');
        $thumbnail_type_name = 'cart_default';
        $preferred_types = ['medium_default', 'home_default', 'small_default'];
        foreach ($preferred_types as $pref_type) {
            foreach ($image_types as $type) {
                if ($type['name'] == $pref_type) {
                    $thumbnail_type_name = $type['name'];
                    break 2;
                }
            }
        }
         if (empty($thumbnail_type_name) && !empty($image_types)) { // Fallback if none of preferred are found
            $thumbnail_type_name = $image_types[0]['name'];
        }


        foreach ($images as $key => $image_data) {
            $image_obj = new Image($image_data['id_image']);
            // Robust link_rewrite handling
            $link_rewrite = '';
            if (is_array($product->link_rewrite) && isset($product->link_rewrite[$this->context->language->id])) {
                $link_rewrite = $product->link_rewrite[$this->context->language->id];
            } elseif (is_array($product->link_rewrite)) {
                $link_rewrite = current($product->link_rewrite);
            } else {
                $link_rewrite = $product->link_rewrite;
            }
            if (empty($link_rewrite)) { // Ultimate fallback for link_rewrite
                $link_rewrite = 'product';
            }


            $image_path_full = _PS_PROD_IMG_DIR_ . $image_obj->getImgPath() . '.' . $image_obj->image_format;
            $dims = file_exists($image_path_full) ? @getimagesize($image_path_full) : null;

            $current_image_data = [
                'id_image' => $image_obj->id,
                'url' => $this->context->link->getImageLink($link_rewrite, $image_obj->id, $thumbnail_type_name),
                'original_url' => $this->context->link->getImageLink($link_rewrite, $image_obj->id),
                'legend' => isset($image_obj->legend[$this->context->language->id]) ? $image_obj->legend[$this->context->language->id] : '',
                'position' => $image_obj->position,
                'cover' => (bool)$image_obj->cover,
                'width' => $dims[0] ?? null,
                'height' => $dims[1] ?? null,
            ];
            $product_images_data[] = $current_image_data;

            // Determine first image details (prefer cover, then first in list)
            if ($image_obj->cover) {
                $first_image_details = $current_image_data;
            } elseif ($key === 0 && $first_image_details === null) {
                 $first_image_details = $current_image_data;
            }
        }
        // If no cover image was found and list is not empty, ensure first_image_details is the actual first one
        if (empty($first_image_details) && !empty($product_images_data)) {
            $first_image_details = $product_images_data[0];
        }


        $defaultParamsJson = Configuration::get(self::DEFAULT_CONFIG_NAME);
        $defaultParams = json_decode($defaultParamsJson, true);
	    $loadedDefaults = $this->getDefaultParams(); // Get the full structure
		if ($defaultParams && is_array($defaultParams)) {
    		$defaultParams = array_merge($loadedDefaults, $defaultParams); // Merge, loaded values take precedence
		} else {
    		$defaultParams = $loadedDefaults; // Fallback to hardcoded defaults
		}

	    $defaultParams['prompt_upsampling'] = (bool)$defaultParams['prompt_upsampling'];
        $defaultParams['upscale_target_specific'] = (bool)$defaultParams['upscale_target_specific'];
        $defaultParams['upscale_face_enhance'] = (bool)$defaultParams['upscale_face_enhance'];

    // Inicializar accountDetails como array vacío
    $accountDetails = [];
    
    // Solo obtener detalles de cuenta si hay API key Y no hay errores de registro
    if (!empty(self::getBflApiKey())) {
        $accountDetailsResponse = $this->getAccountDetails(self::getBflApiKey());
        if (is_array($accountDetailsResponse) && !isset($accountDetailsResponse['error'])) {
            $accountDetails = $accountDetailsResponse;
        }
    }	 
	 
        $this->context->smarty->assign([
            'ai_product_images' => $product_images_data,
            'ai_id_product' => $id_product,
            'ai_controller_url' => $this->context->link->getAdminLink('AdminAiImageEdit'),
            'ai_ajax_token' => Tools::getAdminTokenLite('AdminAiImageEdit'),
            'ai_api_key_bfl' => self::getBflApiKey(),
            'token' => Tools::getAdminTokenLite('AdminProducts'), // For the "Refresh & Go Back" button
            'ai_saved_configs' => self::getSavedConfigsStatic(), // <<< CHANGED TO self::
            'ai_default_params' => $defaultParams,
            'ai_selected_image_initial' => $first_image_details,
			'link' => $this->context->link, 
	        'ai_user_plan' => $accountDetails['plan'] ?? 'freemium', 
            'ai_user_id' => $accountDetails['id'] ?? null,     
			'prestashop_lang_iso_code' => $this->context->language->iso_code,
        ]);

        return $this->display(__FILE__, 'views/templates/admin/product_tab.tpl');
    }
}